/*  File   : chtype.c
    Author : Richard A. O'Keefe
    Updated: 11/14/97
    Purpose: Classify characters.
*/

/*  We want to classify characters as
    "end of file", "letter", "digit", and "other".
    We use a table to convert (unsigned characters) or EOF codes
    such as getc() returns to numeric codes:
   -1      for EOF .. -1
    0 to  9   for '0' to '9'
   10 to 35   for 'a' to 'z'
   10 to 35   for 'A' to 'Z'
   36      for '_'
   >36     for any other characters we allow in identifiers
        (currently there are no such characters).

    We also use this table to recognise letters in a case independent
    way.  Instead of asking whether (c == 'a' || c == 'A'), or asking
    whether (tolower(c) == 'a'), we ask whether (digval(c) == DIGIT_A).

    Formerly, this information was duplicated in two separate tables,
   chtype[]   in main.c
   digval[]   in expr.c

    One thing that is not clear to me is whether it is better to have
    chtype[] as an array of signed char or as an array of int.  Since
    the Alpha has no byte load instruction, and since the table isn't
    too large, I have chosen to make it int for speed.

    If `latin1' is defined when this file is compiled, the accented
    letters in ISO Latin 1 will be accepted as letters like `_'.  So
    far this option has not been tested.

    ourlims.h provides UCHAR_MAX
*/
#include "common.h"
#include "ourlims.h"
#include "chtype.h"

int  chtype[UCHAR_MAX + 1 - EOF];
#ifdef  EXTENDED
unsigned char dncase[UCHAR_MAX + 1];
#else
unsigned char dncase[1];  /* just a dummy */
#endif

static unsigned char digit[]    = "0123456789";
static unsigned char ucletter[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
static unsigned char lcletter[] = "abcdefghijklmnopqrstuvwxyz";

void
init_chtype H0(
    void
)
    {
   int c;
   unsigned char const *p;

   for (c = EOF; c <= UCHAR_MAX; c++)
       chtype[c - EOF] = -1;
   for (c =  0, p = digit;    *p != '\0'; c++, p++) chtype[*p - EOF] = c;
   for (c = 10, p = lcletter; *p != '\0'; c++, p++) chtype[*p - EOF] = c;
   for (c = 10, p = ucletter; *p != '\0'; c++, p++) chtype[*p - EOF] = c;
   chtype['_' - EOF] = 36;
#ifdef  EXTENDED
   for (c = 0; c <= UCHAR_MAX; c++)
       dncase[c] = c;
   for (c = 0; ucletter[c] != '\0'; c++)
       dncase[ucletter[c]] = lcletter[c];
#ifdef  latin1
   for (c = 192; c < 256; c++)
       if (c != 215 && c != 247)
     chtype[c - EOF] = 36;
   for (c = 192; c < 223; c++)
       if (c != 215)
     dncase[c] = c+32;
#endif
#endif
    }

#ifdef  EXTENDED
#ifdef __STDC__
#include <string.h>
#else
extern cmpfn strcmp;
#define const
#endif

cmpfn compare = strcmp;

int
dncmp H2(
    char const *,s,
    char const *,t
)
    {
   unsigned char const *p = (unsigned char const *)s;
   unsigned char const *q = (unsigned char const *)t;
   unsigned char c;

   while (dncase[c = *p] == dncase[*q] && c != '\0') p++, q++;
   return dncase[c] - dncase[*q];
    }
#endif

