/*  File   : divide.c
    Author : Richard A. O'Keefe
    Updated: 11/14/97
    Defines: divide(x, y, 0(div) | 1(rem) | 2(mod))
*/


/*  We want M4 scripts to be portable.
    That means we want expr(...) to be portable.
    And that means we want x div y, x rem y, and x mod y to be portable.
    There are limits to the portability we can get.
    C89 guarantees that long int is _at least_ thirty-two bits worth,
    and that unsigned long int can hold |x| for any x in long int,
    but it does not guarantee that long int is _exactly_ thirty-two
    bits, and there are 36-bit and 48-bit implementations of C,
    as well as the more obvious long=64-bit ones.

    However, as far as I know, Public Domain M4 has been used only on
    the following architectures:
   IBM System/370
   Vax-11
   MC680x0
   NS32x32
   Intel *86 family
   SPARC V7, V8, V9 (32-bit mode)
    and it _should_ give the same results on any 32-bit C system.
    SPARC V9 is 64-bit, Alpha is 64-bit, there are 64-bit Power(PC)s,
    and some consideration should be given to 32/64-bit portability.
    That has not been done yet, though.

    The problem is, there's another portability problem.
    C provides two division operators, / and %, not three,
    and they are not fully defined, in order to give implementors
    freedom to provide whatever is most efficient on there machine.
    I understand that this freedom may be taken away in C9x, but
    the problem remains that / and % are typically the "wrong"
    division and remainder.  That is

   x/y = trunc(x:rational / y:rational)     "div"
   x%y = x - (x/y)*y            "rem"

    The more useful remainder operator is

   x - floor(x/y)*y          "mod"

    The operations provided in M4 are

    (x div y) = (x / y) = trunc(x/y)
    (x rem y) = (x % y) = x - (x div y)*y
    (x mod y) = (x \ y) = y == 0 ? x : x - floor(x/y)*y

    The definition of mod has been extended to cover 0 in accordance
    with Knuth's advice in "Concrete Mathematics".

    We want to deal with abs(x) and abs(y), and it would all be so easy
    except that most C systems use twos-complement arithmetix (a pox on
    it!), and in twos-complement arithmetic, abs() can fail and div can
    overflow.  Neither can happen in sensible arithmetics, sigh.

    C allows three implementations of integer arithmetic:
   Sign-and-magnitude  ~0 == -MAXINT
   Ones-complement     ~0 == -0
   Twos-complement     ~0 == -1
    In order to allow for the extra value, we have to use unsigned long
    for the result of negation.  And we have to put casts in just the
    right spots.  We use the test (~0 == -1) to check for twos-complement.

    There is a version issue with eval(x/0).
    *   pdm4 produces an error message and keeps going,
   returning 0 from the whole expression.
    *   GNU M4 does the same, but it's a more informative message.
    *   System V M4 crashes and dumps core.
    What does _your_ m4 do with modtest.m4?
*/
#ifndef TWOS_COMPLEMENT
#define TWOS_COMPLEMENT (~0 == -1)
#endif
#define uneg(x) (TWOS_COMPLEMENT ? -(unsigned)(x) : (unsigned)-(x))
#define MAX_LONG (long int)(~(unsigned long int)0 >> 1)

#ifdef TEST
#include <stdio.h>
#define experr(x) fprintf(stdout, "m4: %s\n", x)
#else
#include "mdef.h"
#include "extr.h"
#endif

long
divide H3(
    long int,x,
    long int,y,
    int,     c
)
    {
   if (y > 1) {
       unsigned long int uy = y;
       unsigned long int ux = x < 0 ? uneg(x) : (unsigned)x;
       unsigned long int  r;

       switch (c) {
       case 2:                  /*mod*/
     r = ux%uy;
     return x < 0 && r != 0 ? (long int)(uy-r) : (long int)r;
       case 1:                  /*rem*/
           r = ux%uy;
           return x < 0 ? -(long int)r : (long int)r;
       default:
     r = ux/uy;              /*div*/
     return x < y ? -(long int)r : (long int)r;
       }
   } else
   if (y < -1) {
       unsigned long int uy = uneg(y);
       unsigned long int ux = x < 0 ? uneg(x) : (unsigned)x;
       unsigned long int  r;

       switch (c) {
       case 2:                  /*mod*/
     r = ux%uy;
     return x > 0 && r != 0 ? -(long int)(uy-r) : -(long int)r;
       case 1:                  /*rem*/
     r = ux%uy;
     return x < 0 ? -(long int)r : (long int)r;
       default:
     r = ux/uy;
     return x < 0 ? (long int)r : -(long int)r;
       }
   } else
   if (c == 2) {                /*mod*/
       return y == 0 ? x : 0;
   } else
   if (y == 0) {                /*rem/div*/
       experr("division by zero");
       return c == 1 ? x : 0;
   } else
   if (c == 1) {                /*rem*/
       return 0;
   } else                    /*div*/
   if (TWOS_COMPLEMENT && y < 0 && x == -MAX_LONG-TWOS_COMPLEMENT) {
       experr("division overflow");
       return x;
   } else {
       return y < 0 ? -x : x;
   }
    }

#ifdef TEST
int
main H0(
    void
)
    {
   long x, y;

   printf(" x  y x/y x%%y x\\y ? ?\n");
   for (y = -5; y <= 5; y++) {
       for (x = -5; x <= 5; x++) {
     long d = divide(x, y, 0);
     long r = divide(x, y, 1);
     long m = divide(x, y, 2);
     printf("%2d %2d %3d %3d %3d %c %c",
         x, y, d, r, m, "ny"[d*y+r == x],
         "yn"[(y > 0 && m < 0) || (y < 0 && m > 0)]);
     if (y != 0) printf(" [%3d %3d]", x/y, x%y);
     printf("\n");
       }
   }
   return 0;
    }
#endif
