/*  File   : expr.c
    Authors: Mike Lutz & Bob Harper
    Editors: Ozan Yigit & Richard A. O'Keefe
    Updated: 11/19/97
    Purpose: arithmetic expression evaluator.

    expr() performs a standard recursive descent parse to evaluate any
    expression permitted byf the following grammar:

      expr    :       query EOS
      query   :       lor
              |       lor "?" query ":" query
      lor     :       land { "||" land } or OR,  for Pascal
      land    :       bor { "&&" bor }      or AND, for Pascal
      bor     :       bxor { "|" bxor }
      bxor    :       band { "^" band }
      band    :       eql { "&" eql }
      eql     :       relat { eqlop relat }
      relat   :       shift { relop shift }
      shift   :       primary { shftop primary }
      primary :       term { addop term }
      term    :       unary { mulop unary }
      unary   :       factor
              |       unop unary
      factor  :       constant
              |       "(" query ")"
      constant:       num
              |       "'" CHAR "'"    or '"' CHAR '"'
      num     :       DIGIT        full ANSI C syntax
              |       DIGIT num
      eqlop   :       "="
              |       "=="
              |       "!="
      relop   :       "<"       or <>, Pascal not-equal
              |       ">"
              |       "<="         or =<, for Prolog users.
              |       ">="
      shftop  :       "<<"
              |       ">>"
      addop   :       "+"       Note, AND and OR are _not_
         |       "-"         addops; they're like &&, ||
      mulop   :       "*"
              |       "/" | "div"
              |       "%" | "rem"
              |       "\" | "mod"
      unop    :       "!" | "not"
              |       "~"
              |       "-"
              |       "even" | "odd" | "sqr"

    This expression evaluator was lifted from a public-domain
    C Pre-Processor included with the DECUS C Compiler distribution.
    It has been hacked somewhat to be suitable for m4.

     19-Nov-1997 Added the unary operators odd, even, and sqr.
        Made numcom() use a local variable.

     17-Nov-1997 Made experr() report the expression, and added
        GNU's "0b<base 2 number>" numeric constants.

     13-Nov-1997 Added divide() in divide.c so that eval(/),
        eval(%), and eval(\) are portable to all 32-bit
        systems.

     12-Nov-1997 1. Put more detail into the description above.
        2. Added based number support (from the version
        of expr.c that goes with printf.c, elsewhere).
        3. Switched mod and rem, to fit with Ada.

     7-Nov-1997     prototype for abort() added.  Really we should
        include stdlib.h.

    17-Oct-1996     digval merged with chtype at long last.

     8-Sep-1993     The mulop `\' = `rem' was added.  X\Y is always
        positive if Y is.  It is not fully defined if Y
        is negative.

    26-Mar-1993     Changed to work in any of EBCDIC, ASCII, DEC MNCS,
        or ISO 8859/n.

    26-Mar-1993     Changed to use "long int" rather than int, so that
        we get the same 32-bit arithmetic on a PC as on a Sun.
        It isn't fully portable, of course, but then on a 64-
        bit machine we _want_ 64-bit arithmetic...
        Shifting rewritten (using LONG_BIT) to give signed
        shifts even when (long) >> (long) is unsigned.

    26-Mar-1993     I finally got sick of the fact that &&, ||, and ?:
        don't do conditional evaluation.  What is the good
        of having eval(0&&(1/0)) crash and dump core?  Now
        every function has a doit? argument.

    26-Mar-1993     charcon() didn't actually accept 'abcd', which it
        should have.  Fixed it.

    20-Apr-1993     eval(1/0) and eval(1%0) dumped core and crashed.
        This is also true of the System V r 3.2 m4, but
        it isn't good enough for ours!  Changed it so that
        x % 0 => x  as per Concrete Mathematics
        x / 0 => error and return 0 from expr().
*/

#define FALSE   0
#define TRUE  1

#include <stdio.h>
#include <setjmp.h>
static jmp_buf expjump;      /* Error exit point for expr() */

#include "common.h"
#include "ourlims.h"
#include "chtype.h"

extern long int divide P3(long int, long int, int);

static unsigned char const *fstchr;   /* Starting point */
static unsigned char const *nxtchr;   /* Parser scan pointer */

#define isblank(b)  ((unsigned)(b)-(unsigned)1 < (unsigned)' ')
#define deblank0 while (isblank(*nxtchr)) nxtchr++
#define deblank1 while (isblank(*++nxtchr))
#define deblank2 nxtchr++; deblank1


#ifdef  __STDC__
static long int query(int);
#include <stdlib.h>
#else
static long int query();
extern void abort();
#endif


/*  experr(msg)
    prints an error message, resets environment to expr(), and
    forces expr() to return FALSE.
*/
void
experr H1(char const *,msg)
    {
   (void) fprintf(stderr, "m4: %s\n", msg);
   (void) fprintf(stderr, "in: %s\n", (char *)fstchr);
   longjmp(expjump, -1);  /* Force expr() to return FALSE */
    }


/*  numcon recognises all integers.  It has to accomodate a variety
    of styles:

    From C:
   0ooooo is in octal.
   0Xhhhh is in hexadecimal
   decimal numbers must start with 1..9
   unsigned (uU) and long (lL) suffixes, which are ignored,
   with the numbers always being signed long.

    From Ada:
   base#digits# notation and underscores allowed between digits

    From ISO Pascal Extended:
   base#digits notation.

    From Prolog:
   base'digits notation

    From Lisp and Algol 68:
   baseRdigits notation.

    I have never been able to make my mind up whether to accept those
    horrible suffixes all the time, or only when a radix is not present.
    In November 1997, I decided _not_ to accept suffixes when a radix
    is present.  So now the grammar is

    <numcon> ::= <num> <suffix>
    |   <num> <radix char> <ext digit>+ [<radix char>]

    <radix char> ::= ' | # (must be the same at each end)

    <num> ::= 0 {<octit> | _}*
      |  {0x | 0X} {<hexit> | _}*
      |  <1..9> {<digit> | _}*

    <suffix> ::= {u | U | l | L}*
*/
static long int
numcon H1(int,doit)
    {
   register unsigned char const *n = nxtchr;
   register long int v;   /* current value */
   register int c;     /* character or digit value */
   register int b;     /* base (radix) */

   if (!doit) {
       do c = *n++; while (digval(c) >= 0);
       if (c == '#' || c == '\'' || digval(c) == DIGIT_R) {
     int q = c;
     do c = *n++; while (digval(c) >= 0);
     if (c == q) c = *n++;
       }
       while (isblank(c)) c = *n++;
       n--;      /* undo lookahead */
       nxtchr = n;
       return 0;
   }

   v = digval(*n++);   /* We already know it's a digit */
   if (v != 0) {
       b = 10;      /* decimal number */
   } else
   if (digval(*n) == DIGIT_X) {
       n++;
       b = 16;      /* hexadecimal number */
   } else
   if (digval(*n) == DIGIT_O) {
       n++;
       b = 8;    /* octonary number */
   } else
   if (digval(*n) == DIGIT_B) {
       n++;
       b = 2;    /* binary number */
   } else {
       b = 8;    /* octonary number */
   }
   do {
       while ((unsigned)digval(c = *n++) < b)
     v = v*b + digval(c);
   } while (c == '_');
   if (v <= 36 && v >= 2 &&
       (c == '#' || c == '\'' || digval(c) == DIGIT_R)
   ) {
       int q = c;
       b = v;
       v = 0;
       do {
     while (digval(c = *n++) < b)
         v = v*b + digval(c);
       } while (c == '_');
       if (c == q) c = *n++;
   } else {
       /* skip suffix */
       while (digval(c) == DIGIT_L || digval(c) == DIGIT_U)
     c = *n++;
   }
   while (isblank(c)) c = *n++;
   n--;    /* unread c */
   nxtchr = n;
   return v;
    }


/*  <charcon> ::= <qt> { <char> } <qt>
    Note: multibyte constants are accepted.
    Note: BEL (\a) and ESC (\e) have the same values in EBCDIC and ASCII.
*/
static long int
charcon H1(int,doit)
    {
   register int i;
   long int value;
   register int c;
   int q;
   int v[sizeof value];

   q = *nxtchr++;      /* the quote character */
   for (i = 0; ; i++) {
       c = *nxtchr++;
       if (c == q) {   /* end of literal, or doubled quote */
     if (*nxtchr != c) break;
     nxtchr++;   /* doubled quote stands for one quote */
       }
       if (i == (int)sizeof value)
     experr("Unterminated character constant");
       if (c == '\\') {
     switch (c = *nxtchr++) {
         case '0': case '1': case '2': case '3':
         case '4': case '5': case '6': case '7':
        c -= '0';
        if ((unsigned)(*nxtchr - '0') < 8)
            c = (c << 3) | (*nxtchr++ - '0');
        if ((unsigned)(*nxtchr - '0') < 8)
            c = (c << 3) | (*nxtchr++ - '0');
        break;
         case 'n': case 'N': c = '\n'; break;
         case 'r': case 'R': c = '\r'; break;
         case 't': case 'T': c = '\t'; break;
         case 'b': case 'B': c = '\b'; break;
         case 'f': case 'F': c = '\f'; break;
         case 'a': case 'A': c = 007;  break;
         case 'e': case 'E': c = 033;  break;
#if  ' ' == 64
         case 'd': case 'D': c = 045;  break; /*EBCDIC DEL */
#else
         case 'd': case 'D': c = 127;  break; /* ASCII DEL */
#endif
         default :          break;
     }
       }
       v[i] = c;
   }
   deblank0;
   if (!doit) return 0;
   for (value = 0; --i >= 0; ) value = (value << CHAR_BIT) | v[i];
   return value;
    }


/*  <unary> ::= <unop> <unary> | <factor>
    <unop> ::= '!' || '~' | '-' | 'not'
    <factor> ::= '(' <query> ')' | <'> <char> <'> | <"> <char> <"> | <num>
*/
static long int
unary H1(int,doit)
    {
   long int v;

   switch (nxtchr[0]) {
       case 'e': case 'E':
        if (digval(nxtchr[1]) != DIGIT_V
        ||  digval(nxtchr[2]) != DIGIT_E
        ||  digval(nxtchr[3]) != DIGIT_N)
            experr("Bad 'even'");
        nxtchr += 4; deblank0;
        return 1&~unary(doit);
       case 'o': case 'O':
        if (digval(nxtchr[1]) != DIGIT_D
        ||  digval(nxtchr[2]) != DIGIT_D)
            experr("Bad 'odd'");
        nxtchr += 3; deblank0;
        return 1&unary(doit);
       case 's': case 'S':
        if (digval(nxtchr[1]) != DIGIT_Q
        ||  digval(nxtchr[2]) != DIGIT_R)
            experr("Bad 'sqr'");
        nxtchr += 3; deblank0;
        v = unary(doit);
        return v*v;
       case 'n': case 'N':
        if (digval(nxtchr[1]) != DIGIT_O
        ||  digval(nxtchr[2]) != DIGIT_T)
            experr("Bad 'not'");
        nxtchr += 2;
        /*FALLTHROUGH*/
       case '!': deblank1; return !unary(doit);
       case '~': deblank1; return ~unary(doit);
       case '-': deblank1; return -unary(doit);
       case '+': deblank1; return  unary(doit);
       case '(': deblank1; v = query(doit);
        if (nxtchr[0] != ')') experr("Bad factor");
        deblank1; return v;
       case '\'':
       case '\"':   return charcon(doit);
       case '0': case '1': case '2':
       case '3': case '4': case '5':
       case '6': case '7': case '8':
       case '9': return numcon(doit);
       default :   experr("Bad constant");
   }
   return 0;  /*NOTREACHED*/
    }


/*  <term> ::= <unary> { <mulop> <unary> }
    <mulop> ::= '*' | '/' | '%' | '\'
                    | div | rem | mod

    Division is handled by the (separate) function divide, in order
    to ensure maximum portability.
*/
static long int
term H1(int,doit)
    {
   register long int vl, vr;

   vl = unary(doit);
   for (;;)
       switch (nxtchr[0]) {
     case '*':
         deblank1;
         vr = unary(doit);
         if (doit) vl *= vr;
         break;
     case 'd': case 'D':
         if (digval(nxtchr[1]) != DIGIT_I
         ||  digval(nxtchr[2]) != DIGIT_V)
        experr("Bad 'div'");
         nxtchr += 2;
         /*FALLTHROUGH*/
     case '/':
         deblank1;
         vr = unary(doit);
         if (doit) vl = divide(vl, vr, 0);
         break;
     case 'r': case 'R':
         if (digval(nxtchr[1]) != DIGIT_E
         ||  digval(nxtchr[2]) != DIGIT_M)
        experr("Bad 'rem'");
         nxtchr += 2;
         /*FALLTHROUGH*/
     case '%':
         deblank1;
         vr = unary(doit);
         if (doit) vl = divide(vl, vr, 1);
         break;
     case 'm': case 'M':
         if (digval(nxtchr[1]) != DIGIT_O
         ||  digval(nxtchr[2]) != DIGIT_D)
        experr("Bad 'mod'");
         nxtchr += 2;
         /*FALLTHROUGH*/
     case '\\':
         deblank1;
         vr = unary(doit);
         if (doit) vl = divide(vl, vr, 2);
         break;
     default:
         return vl;
       }
   /*NOTREACHED*/
    }

/*  <primary> ::= <term> { <addop> <term> }
    <addop> ::= '+' | '-'
*/
static long int
primary H1(int,doit)
    {
   register long int vl;

   vl = term(doit);
   for (;;) {
       if (nxtchr[0] == '+') {
     deblank1;
     if (doit) vl += term(doit); else (void)term(doit);
       } else
       if (nxtchr[0] == '-') {
     deblank1;
     if (doit) vl -= term(doit); else (void)term(doit);
       } else {
     return vl;
       }
   }
   /*NOTREACHED*/
    }


/*  <shift> ::= <primary> { <shftop> <primary> }
    <shftop> ::= '<<' | '>>'
*/
static long int
shift H1(int,doit)
    {
   register long int vl, vr;

   vl = primary(doit);
   for (;;) {
       if (nxtchr[0] == '<' && nxtchr[1] == '<') {
     deblank2;
     vr = primary(doit);
       } else
       if (nxtchr[0] == '>' && nxtchr[1] == '>') {
     deblank2;
     vr = -primary(doit);
       } else {
     return vl;
       }
       /* The following code implements shifts portably */
       /* Shifts are signed shifts, and the shift count */
       /* acts like repeated one-bit shifts, not modulo anything */
       if (doit) {
     if (vr >= LONG_BIT) {
         vl = 0;
     } else
     if (vr <= -LONG_BIT) {
         vl = -(vl < 0);
     } else
     if (vr > 0) {
         vl <<= vr;
     } else
     if (vr < 0) {
         vl = (vl >> -vr) | (-(vl < 0) << (LONG_BIT + vr));
     }
       }
   }
   /*NOTREACHED*/
    }


/*  <relat> ::= <shift> { <relop> <shift> }
    <relop> ::= '<=' | '>=' | '=<' | '=>' | '<' | '>' | '<>'
    Here I rely on the fact that '<<' and '>>' are swallowed by <shift>
*/
static long int
relat H1(int,doit)
    {
   register long int vl;

   vl = shift(doit);
   for (;;)
       switch (nxtchr[0]) {
     case '=':
         switch (nxtchr[1]) {
        case '<':         /* =<, take as <= */
            deblank2;
            vl = vl <= shift(doit);
            break;
        case '>':         /* =>, take as >= */
            deblank2;
            vl = vl >= shift(doit);
            break;
        default:       /* == or =; OOPS */
            return vl;
         }
         break;
     case '<':
         if (nxtchr[1] == '=') {      /* <= */
        deblank2;
        vl = vl <= shift(doit);
         } else
         if (nxtchr[1] == '>') {      /* <> (Pascal) */
        deblank2;
        vl = vl != shift(doit);
         } else {            /* < */
        deblank1;
        vl = vl < shift(doit);
         }
         break;
     case '>':
         if (nxtchr[1] == '=') {      /* >= */
        deblank2;
        vl = vl >= shift(doit);
         } else {            /* > */
        deblank1;
        vl = vl > shift(doit);
         }
         break;
     default:
         return vl;
   }
   /*NOTREACHED*/
    }


/*  <eql> ::= <relat> { <eqlop> <relat> }
    <eqlop> ::= '!=' | '==' | '='
*/
static long int
eql H1(int,doit)
    {
   register long int vl;

   vl = relat(doit);
   for (;;)
       if (nxtchr[0] == '!' && nxtchr[1] == '=') {
     deblank2;
     vl = vl != relat(doit);
       } else
       if (nxtchr[0] == '=' && nxtchr[1] == '=') {
     deblank2;
     vl = vl == relat(doit);
       } else
       if (nxtchr[0] == '=') {
     deblank1;
     vl = vl == relat(doit);
       } else
     return vl;
   /*NOTREACHED*/
    }


/*  <band> ::= <eql> { '&' <eql> }
*/
static long int
band H1(int,doit)
    {
   register long int vl;

   vl = eql(doit);
   while (nxtchr[0] == '&' && nxtchr[1] != '&') {
       deblank1;
       if (doit) vl &= eql(doit); else (void)eql(doit);
   }
   return vl;
    }


/*  <bxor> ::= <band> { '^' <band> }
*/
static long int
bxor H1(int,doit)
    {
   register long int vl;

   vl = band(doit);
   while (nxtchr[0] == '^') {
       deblank1;
       if (doit) vl ^= band(doit); else (void)band(doit);
   }
   return vl;
    }


/*  <bor> ::= <bxor> { '|' <bxor> }
*/
static long int
bor H1(int,doit)
    {
   register long int vl;

   vl = bxor(doit);
   while (nxtchr[0] == '|' && nxtchr[1] != '|') {
       deblank1;
       if (doit) vl |= bxor(doit); else (void)bxor(doit);
   }
   return vl;
    }


/*  <land> ::= <bor> { '&&' <bor> }
*/
static long int
land H1(int,doit)
    {
   register long int vl;

   vl = bor(doit);
   for (;;) {
       if (nxtchr[0] == '&') {
     if (nxtchr[1] != '&') break;
     deblank2;
       } else
       if (digval(nxtchr[0]) == DIGIT_A) {
     if (digval(nxtchr[1]) != DIGIT_N) break;
     if (digval(nxtchr[2]) != DIGIT_D) break;
     nxtchr += 2; deblank1;
       } else {
     /* neither && nor and */
     break;
       }
       vl = bor(doit && vl != 0) != 0;
   }
   return vl;
    }


/*  <lor> ::= <land> { '||' <land> }
*/
static long int
lor H1(int,doit)
    {
   register long int vl;

   vl = land(doit);
   for (;;) {
       if (nxtchr[0] == '|') {
     if (nxtchr[1] != '|') break;
       } else
       if (digval(nxtchr[0]) == DIGIT_O) {
     if (digval(nxtchr[1]) != DIGIT_R) break;
       } else {
     /* neither || nor or */
     break;
       }
       deblank2;
       vl = land(doit && vl == 0) != 0;
   }
   return vl;
    }


/*  <query> ::= <lor> [ '?' <query> ':' <query> ]
*/
static long int
query H1(int,doit)
    {
   register long int test, true_val, false_val;

   test = lor(doit);
   if (*nxtchr != '?') return test;
   deblank1;
   true_val = query(doit && test != 0);
   if (*nxtchr != ':') experr("Bad query");
   deblank1;
   false_val = query(doit && test == 0);
   return test != 0 ? true_val : false_val;
    }


long int
expr H1(char const *,expbuf)
    {
   register long int rval;

   nxtchr = (unsigned char const *)expbuf;
   deblank0;
   fstchr = nxtchr;
   if (setjmp(expjump) != 0) return FALSE;
   rval = query(TRUE);
   if (*nxtchr != '\0') {
       experr("Ill-formed expression");
   }
   return rval;
    }

