/*  File   : look.c
    Author : Ozan Yigit
    Updated: 11/19/97 (4 May 1992)
    Purpose: Hash table for M4
*/

#include "mdef.h"
#include "extr.h"
#include "chtype.h"

ndptr hashtab[HASHSIZE];


/*  hash(name)
    compute hash value using the proverbial hashing function.
    Taken from Kernighan & Ritchie.
*/
static int
hash H1(char const *,name)
    {
   register unsigned char const *n = (unsigned char const *)name;
   register unsigned int         h = 0;
   register unsigned char c;

   if (compare == strcmp) {
       while ((c = *n++) != EOS) HUPDATE(h, c);
   } else {
       while ((c = *n++) != EOS) HUPDATE(h, dncase[c]);
   }
   return (int)(h%HASHSIZE);
    }

/*  lookup(name)
    find name in the hash table
*/
ndptr
lookup H1(char const *,name)
    {
   register ndptr p;

   for (p = hashtab[hash(name)]; p != nil; p = p->nxtptr)
       if (compare(name, p->name) == 0)
     break;
   return p;
    }

/*  addent(name)
    hash and create an entry in the hash table.
    The new entry is added at the front of a hash bucket.
    BEWARE: the type field is UNDEFINED.
*/
ndptr
addent H1(char const *,name)
    {
   register ndptr p, *h;

   p = (ndptr)malloc(sizeof *p);
   if (p == NULL) error1(memsg);
   h = &hashtab[hash(name)];
   p->name = dupstr(name);
   p->defn = null;
   p->nxtptr = *h;
   *h = p;
   return p;
    }


/*  addkywd(prefix, name, type)
    stores a keyword in the hash table.
    There is a restriction:  strlen(prefix) + strlen(name) must be
    no more than 80.  This is an arbitrary but reasonable restriction.
    This function is only called at startup, and only used to install
    a fixed set of built-in functions.
*/
void
addkywd H3(
    char const *,prefix,
    char const *,name,
    int         ,type
)
    {
   char fullname[81];
   char buff[5];
   register ndptr p;
   int pl = strlen(prefix);

   if (pl + strlen(name) >= sizeof fullname)
       error2("-m prefix too long:", prefix);
   (void) memcpy(fullname, prefix, pl);
   (void) strcpy(fullname+pl, name);
   p = addent(fullname);
   if (type == MACRTYPE) {
       p->defn = null;
   } else {
       buff[0] = ARGFLAG;
       buff[1] = BIMCHAR;
       buff[2] = type/10 + '0';
       buff[3] = type%10 + '0';
       buff[4] = EOS;
       p->defn = dupstr(buff);
   }
   p->type = type;
    }


/*  remhash(name, all)
    remove one entry (all==0) or all entries (all!=0) for a given name
    from the hash table.  All hash table entries must have been obtained
    from malloc(), so it is safe to free the records themselves.
    Once upon a time, the ->name field could have been a static literal,
    passed to addkywd.  Two changes have taken place:  built-ins now have
    a definition string ($!mn) so that define(foo,defn(`bar')) will
    work, and built-in names are dynamically copied too, so that a GNU-ish
    prefix can be applied when they are installed.  Accordingly, the old
    STATIC flag has gone away, and _all_ macro names without exception
    are dynamically allocated, and _all_ macro definitions are either
    null (a static empty string) or dynamically allocated.  So it is now
    safe to free them in more cases.
*/
void
remhash H2(char const *,name, int,all)
    {
   register ndptr p, *h;
   /*  h always points to the pointer to p  */

   h = &hashtab[hash(name)];
   while ((p = *h) != nil) {
       if (compare(p->name, name) == 0) {
     *h = p->nxtptr;         /* delink this record */
     free(p->name);       /* free name (now safe) */
     if (p->defn != null)    /* only free defn if it */
         free(p->defn);      /* is not static empty */
     free(p);       /* free the record itself */
     if (!all) return;    /* first occurrence has gone */
       } else {
     h = &(p->nxtptr);
       }
   }
    }

