/*  File   : main.c
    Author : Ozan Yigit
    Updated: 11/20/97
    Defines: M4 macro processor.
*/

#include "mdef.h"
#include "extr.h"
#include "ourlims.h"
#include "chtype.h"
#include "os.h"
#include <signal.h>

/*
 * m4 - macro processor
 *
 * PD m4 is based on the macro tool distributed with the software
 * tools (VOS) package, and described in the "SOFTWARE TOOLS" and
 * "SOFTWARE TOOLS IN PASCAL" books. It has been expanded to include
 * most of the command set of SysV m4, the standard UN*X macro processor.
 *
 * Since both PD m4 and UN*X m4 are based on SOFTWARE TOOLS macro,
 * there may be certain implementation similarities between
 * the two. The PD m4 was produced without ANY references to m4
 * sources.
 *
 * References:
 *
 * Software Tools distribution: macro
 *
 * Kernighan, Brian W. and P. J. Plauger, SOFTWARE
 * TOOLS IN PASCAL, Addison-Wesley, Mass. 1981
 *
 * Kernighan, Brian W. and P. J. Plauger, SOFTWARE
 * TOOLS, Addison-Wesley, Mass. 1976
 *
 * Kernighan, Brian W. and Dennis M. Ritchie,
 * THE M4 MACRO PROCESSOR, Unix Programmer's Manual,
 * Seventh Edition, Vol. 2, Bell Telephone Labs, 1979
 *
 * System V man page for M4
 *
 * Modification History:
 *
 * Nov 20 1997 RAOK Added m4getenv(Var[,Default]).
 *      m4getenv(var,def) is like ${var-def} in sh(1).
 *      m4getenv(var)     is like ${var?}    in sh(1)
 *      m4getenv(var,)    is like ${var}     in sh(1).
 *
 *      Also fixed a misc.c bug that didn't allow the
 *      output of include(|cmd) to contain another such
 *      form; now uses tmpnam() instead of m4temp, which
 *      should be ok as tmpnam() is in the standard.
 *
 * Nov 19 1997 RAOK Revamped inclusion.  Replaced the doincl()
 *      function with a new incopen() one, and a bunch
 *      of in-line code with a call to inclose().
 *      These new functions are in misc.c.  The big
 *      thing about them is that they handle
 *          -    stdin, and
 *          |command
 *      and that this is done _everywhere_:  in the
 *      include(), sinclude(), paste(), and spaste()
 *      macros, and in processing rhe command line.
 *      It was also revised so that system() is
 *      always available but reports an error if
 *      called on a system that doesn't support it.
 *
 *      It used to be that include() could not be used
 *      in an argument of a macro call.  I have no idea
 *      why.  I have deleted the offending test which
 *      caused the program to abort _after_ doing the
 *      inclusion.
 *
 * Nov 17 1997 RAOK.   Finally implemented dollar-in-macro-name.
 *      This is for my modules-in-Pascal stuff.  The
 *      idea is that TAG will be replaced by the
 *      module instance name, so e.g. TAG$free()
 *      might become stg$free.  In conventional m4,
 *      that's where it stops.  stg$free cannot be
 *      a macro.  But it would be so useful if it
 *      could be.  To get this behaviour, specify
 *      the INFIXCHAR at compile time.
 *
 *      Renamed error() -> error1() and added error2().
 *      Used error2() where I could.
 *
 * Nov 14 1997 RAOK 1. Brought the common.h header in from another
 *      project so that standard C compilers can see
 *      new-style function headings while Class C ones
 *      can see old-style function headings.  Got a wee
 *      bit more lint out, and used 'const' a lot.
 *      2. Renamed __FILE__ to __file__ to match GNU M4
 *      and made it conditional on +EXTENDED, not -NO__FILE.
 *
 * Nov 13 1997 RAOK 1. Added divide() in divide.c so that eval(/),
 *      eval(%), and eval(\) are portable to all 32-bit
 *      systems.
 *                  2.  translit() was a neat piece of work, but
 *      other versions of m4 now do something different,
 *      so it had to change.  Now there is an argument
 *      saying whether to iterate or not.  Also, the
 *      old version had a bug:  if |from| > |to| > 0,
 *      it would address off the end of to.  Fixed.
 *      Also, there is no longer any requirement that
 *      len(first argument) < MAXTOK; the argument is
 *      overwritten in place and any size will do.
 *
 * Nov 11 1997 RAOK 1. Remove the changeparen macro and all support
 *      code for it.  Chanageparen was never finished.
 *      There was just enough of it there to make trouble
 *      (makefile processing and compatibility problems).
 *      It was the wrong approach anyway.
 *      2. Fixed a memory leak.
 *      3. The m4wrap processing used to do pushback(EOF)
 *      which made no sense at all.  There is no portable
 *      way of getting an empty input stream, so the GPBC
 *      code (moved here from a header) now checks for a
 *      null stream.  I hope that replacing input[ilevel]
 *      by ifile will be enough of an efficiency gain to
 *      offset these extra tests.
 *      4. Made ifdef() variadic;
 *      ifdef(name1, body1, ..., nameN, bodyN, default)
 *      evaluates the first bodyI for which nameI is defined
 *      as a macro name, or default if none of them is.
 *      This change was independently motivated and conceived,
 *      but turns out to be necessary for System V Release 4
 *      compatibility, whose ifdef does exactly this.
 *      5. Eliminated a really loathesome cast.  We used to
 *      take an array of unions and cast it to (char**), and
 *      there really isn't any good reason to expect that to
 *      be portable.  main.c and serv.c have been changed to
 *      make argv pointer to stae, not pointer to char*, and
 *      the new macro arg(n) (= argv[n].sstr) has been used
 *      to keep the code readable.  This was actually quite
 *      an easy change, about 15 minutes to code and test.
 *      6. Made dumpdef() report quoters as {quoter} instead
 *      of doing something stupid.
 *      7. The big one:  make changequote() and changecom()
 *      accept multicharacter strings and make them actually
 *      work.  prstr() and strsave() were added for this.
 *
 * Nov 11 1997 RAOK Removed $( $, $) from expand(),
 *      removed the lparen, rparen, and comma variables,
 *      removed the changeparen macro and all support code.
 *      This was for a change that was never finished;
 *      just enough of it was there to cause compatibility
 *      problems.  It was the wrong approach anyway; (,)
 *      should be properties of individual macros, not
 *      global variables.
 *
 * Nov  7 1997 RAOK Made define(foo, defn(`bar')) work even when
 *      bar is a built in macro, by giving built in
 *      macros definition strings of the form ^Dmn^D
 *      where mn is the macro number in decimal.  This
 *      sets a limit of 99 built in macros; if that's
 *      ever a problem we can use hex.  The old STATIC
 *      flag has gone.
 *
 * Nov  7 1996 RAOK Added $[...?...$] for checking argument count.
 *
 * Oct 17 1996 RAOK Merged chtype[] and digval[] at long last.
 *
 * Oct 10 1996 RAOK 1.  Fixed the comment that used to mention $(...$)
 *                      to correctly mention $[...$] instead.
 *      2.  Made extended dosub() 8-bit clean.
 *      3.  Fixed pbqtd() to insert vquote for strings
 *      containing mismatched quotes.
 *      4.  Added prqtd() function.
 *      5.  Made dodump() call prqtd().
 *
 * Sep  9 1993 RAOK 1.  Added $( $, $) $< $\ $> ${ $} in expand() so
 *      that macro bodies can generate the lparen, comma,
 *      rparen, lquote, vquote, rquote, scommt, & ecommt
 *      characters (some day, strings) that are in force
 *      at _expansion_ time rather than definition time.
 *      2.  Made all the new features in expand() depend
 *      on the EXTENDED flag.
 *      3.  $* and $@ now generate whatever "comma" is at
 *      expansion time, rather than a hard-wired ','.
 *
 * Sep  8 1993 RAOK 1.  Added the changeparen(L,R,C) macro, with the
 *      function dochp and variables lparen, rparen, and
 *      and comma that support it.  As yet this does not
 *      affect the rest of the program.
 *      2.  Added the multiplying operator \ (alias 'rem')
 *      to expr().  This is in fact the "mathematician's
 *      mod".  For example, (-2)\3 = 1, (-1)\3 = 2.
 *      3.  Revised the definition of substr() so that it
 *      acts as if the string argument is embedded in an
 *      infinite background.  If EXTENDED is not defined
 *      when PD M4 is compiled, the behaviour remains
 *      compatible with System V.
 *      4.  Fixed a bug concerning diversions.  We map
 *      out-of-range diversions to number 0, which is all
 *      very well and good, but the divnum macro must get
 *      the _actual_ parameter given to divert().  The new
 *      variable real_divnum holds that now.
 *      5.  What I was _really_ trying to achieve was to
 *      get the new $([offset]{@*#}[length]$) construct
 *      working.  It's almost right now, but needs to
 *      handle out-of-range parameters like substr().
 *
 * May 12 1993 RAOK 1.  The eval() macro had somehow acquired an expr()
 *      alias.  Deleted that.
 *      2.  Made the system tests (unix,vms) #ifdef instead
 *      of #if, and added a similar `msdos' macro, conditional
 *      on #ifdef msdos.  Note that "tcc -Dmsdos" #defines
 *      msdos to be _empty_, not 1.  syscmd() and sysval() are
 *      now available for msdos too, where they do work.
 *
 * Mar 26 1993 RAOK 1.  Eliminated magic numbers 8, 255, 256 in favour
 *      of the standard limits CHAR_BIT, UCHAR_MAX, which
 *      are in the new header ourlims.h.  This is part of
 *      the "8-bit-clean M4" project.  To the best of my
 *      belief, all of the code should work in EBCDIC,
 *      ASCII, DEC MNCS, ISO 8859/n, or the Mac character
 *      set, as long as chars are unsigned.  There are
 *      still some places where signed bytes can cause
 *      trouble.
 *
 *      2.  Changed expr() to use long int rather than int.
 *      This is so that we'd get 32-bit arithmetic on a Sun,
 *      Encore, PC, Mac &c.  As part of this, the code for
 *      shifts has been elaborated to yield signed shifts
 *      on all machines.  The charcon() function didn't work
 *      with multi-character literals, although it was meant
 *      to.  Now it does.  pbrad() has been changed so that
 *      eval('abcd',0) => abcd, not dcba, which was useless.
 *
 *      3.  I finally got sick of the fact that &&, ||, and
 *      ?: always evaluate all their arguments.  This is
 *      consistent with UNIX System V Release 3, but I for
 *      one don't see anything to gain by having eval(0&&1/0)
 *      crash when it would simply yield 0 in C.  Now these
 *      operators are more consistent with the C preprocessor.
 *
 * Nov 13 1992 RAOK Added the quoter facility.  The purpose of this is
 *      to make it easier to generate data for a variety of
 *      programming languages, including sh, awk, Lisp, C.
 *      There are two holes in the implementation:  dumpdef
 *      prints junk and undefine doesn't release everything.
 *      This was mainly intended as a prototype to show that
 *      it could be done.
 *
 * Jun 16 1992 RAOK Added vquote and gave changequote a 3rd argument.
 *      The idea of this is to make it possible to quote
 *      ANY string, including one with unbalanced ` or '.
 *      I also made eval(c,0) convert decimal->ASCII, so
 *      that eval(39,0) yields ' and eval(96,0) yields `.
 *
 * Apr 28 1992 RAOK Used gcc to find and fix ANSI clashes, so that
 *      PD M4 could be ported to MS-DOS (Turbo C 3).
 *      Main known remaining problem:  use of mktemp().
 *      Also, command line handling needs to be worked out.
 *
 * Mar 26 1992 RAOK PD M4 now accepts file names on the command line
 *      just like UNIX M4.  Warning:  macro calls must NOT
 *      cross file boundaries.  UNIX M4 doesn't mind;
 *      (m4 a b c) and (cat a b c | m4) are just the same
 *      except for error messages.  PD M4 will report an
 *      unexpected EOF if a file ends while a macro call or
 *      string is still being parsed.  When there is one
 *      file name argument, or none, you can't tell the
 *      difference, and that's all I need.
 *
 * May 15 1991 RAOK DIVNAM was a string constant, but was changed!
 *      Fixed that and a couple of other things to make
 *      GCC happy.  (Also made "foo$bar" get through.)
 *
 * Apr 17 1991 RAOK There was a major mistake.  If you did
 *      define(foo, `1 include(bar) 2') where
 *      file bar held "-bar-" you would naturally
 *      expect "1 -bar- 2" as the output, but you
 *      got "1  2-bar-".  That is, include file
 *      processing was postponed until all macros
 *      had been expanded.  The macro GPBC() was
 *      at fault.  I added bb, bbstack[], and the
 *      code in main.c and serv.c that maintains
 *      them, in order to work around this bug.
 *
 * Apr 12 1991 RAOK inspect() didn't handle overflow well.
 *      Added the automatically maintained macro
 *      __file__, just as in C, if you compile
 *      with EXTENDED set.  At some point, $# had
 *      been made to return a value that was off
 *      by one; it now agrees with SysV M4.
 *
 * Aug 13 1990 RAOK The System V expr() has three arguments:
 *      expression [, radix:10 [, mindigits: 1]]
 *      Brought in my int2str() and wrote pbrad()
 *      to make this work here.  With the wrong #
 *      of args, acts like System V.
 *
 * Aug 11 1990 RAOK Told expr.c about the Pascal operators
 *      not, div, mod, and, or
 *      so that Pascal constant expressions could
 *      be evaluated.  (It still doesn't handle
 *      floats.)  Fixed a mistake in 'character's.
 *
 * Apr 23 1988 RAOK Sped it up, mainly by making putback() and
 *      chrsave() into macros.
 *      Finished the -o option (was half done).
 *      Added the System V -e (interactive) option.
 *
 * Jan 28 1986 Oz   Break the whole thing into little
 *      pieces, for easier (?) maintenance.
 *
 * Dec 12 1985 Oz   Optimize the code, try to squeeze
 *      few microseconds out.. [didn't try very hard]
 *
 * Dec 05 1985 Oz   Add getopt interface, define (-D),
 *      undefine (-U) options.
 *
 * Oct 21 1985 Oz   Clean up various bugs, add comment handling.
 *
 * June 7 1985 Oz   Add some of SysV m4 stuff (m4wrap, pushdef,
 *      popdef, decr, shift etc.).
 *
 * June 5 1985 Oz   Initial cut.
 *
 * Implementation Notes:
 *
 * [1]  PD m4 uses a different (and simpler) stack mechanism than the one
 * described in Software Tools and Software Tools in Pascal books.
 * The triple stack nonsense is replaced with a single stack containing
 * the call frames and the arguments. Each frame is back-linked to a
 *   previous stack frame, which enables us to rewind the stack after
 *   each nested call is completed. Each argument is a character pointer
 * to the beginning of the argument string within the string space.
 * The only exceptions to this are (*) arg 0 and arg 1, which are
 *   the macro definition and macro name strings, stored dynamically
 * for the hash table.
 *
 *     .                  .
 * |   .   |  <-- sp         |  .  |
 * +-------+           +-----+
 * | arg 3 ------------------------------->| str |
 * +-------+           |  .  |
 * | arg 2 --------------+         .
 * +-------+        |
 *     *            |        |     |
 * +-------+        |     +-----+
 * | plev  |  <-- fp     +---------------->| str |
 * +-------+           |  .  |
 * | type  |              .
 * +-------+
 * | prcf  -----------+      plev: paren level
 * +-------+        |     type: call type
 * |   .   |     |     prcf: prev. call frame
 *     .            |
 * +-------+     |
 * | <----------+
 * +-------+
 *
 * [2]  We have three types of null values:
 *
 *   nil  - nodeblock pointer type 0
 *   null - null string ("")
 *   NULL - Stdio-defined NULL
 *
 */

static char *prefix = ""; /* prefix for built-ins        */
char buf[BUFSIZE];     /* push-back buffer         */
char *bp = buf;     /* first available character   */
char *bb = buf;        /* buffer beginning            */
char *endpbb = buf+BUFSIZE;  /* end of push-back buffer     */
stae mstack[STACKMAX+1];  /* stack of m4 machine         */
static
char strspace[STRSPMAX+1];   /* string space for evaluation */
char *ep = strspace;      /* first free char in strspace */
char *endest= strspace+STRSPMAX;/* end of string space         */
int sp;          /* current m4  stack pointer   */
int fp;          /* m4 call frame pointer       */
char *bbstack[MAXINP];    /* stack where bb is saved     */
char *inname[MAXINP];     /* input file name stack       */
FILE *infile[MAXINP];     /* input file stack (0=stdin)  */
FILE *outfile[MAXOUT];    /* diversion array(0=bitbucket)*/
FILE *active;       /* active output file pointer  */
FILE *ifile;        /* current input file          */
static
int realdiv = 0;    /* "real" diversion number     */
int ilevel = -1;       /* input file stack pointer    */
int oindex = 0;     /* diversion index..        */
char *null = "";                /* as it says.. just a null..  */
static char *m4wraps;     /* m4wrap string default..     */
static
int strip = 0;         /* throw away comments?        */
int sysval = 0;        /* sysval() is always defined  */

/*  GPBC() reads a possibly pushed-back character.
    The possibilities are
    - there are pushed back characters (bp != bb)
    - there is an open input file (ifile != NULL)
    - there is nothing left to read (otherwise).
*/
#define GPBC (bp == bb ? (ifile == NULL ? EOF : getc(ifile)) : *--bp)

/*  Quoted strings have the form
    <string> ::= <lquote> <item>* <rquote>
    <item>   ::= <vquote> <character>
              |  <string>
              |  <any character not matching lquote, rquote, or vquote>
    Comments have the form
    <comment> ::= <scomment> <item>* <ecommt>
    <item>    ::= <xcommt> <character>
          |  <any character not matching ecomment or xcommet>

    They have so much in comment that there ought to be a single
    mechanism for them both.  What's more, when processing C++ or C9x,
    we want to deal with as many as three different kinds of comments
    in the same program (slash star, slash slash, and sharp).
*/
char dfltchrs[] = {
    '`',         EOS,     /* left quote string    [`] */
    '\'',        EOS,     /* right quote string      ['] */
    'V'&(' '-1), EOS,     /* verbatim quote string       [^V] */
    '#',         EOS,     /* comment start string    [#] */
    '\n',        EOS,     /* comment end string          [\n] */
    EOS,         EOS,     /* comemnt escape string       [\0] */
};
char *lquote = LQUOTE;
char *rquote = RQUOTE;
char *vquote = VQUOTE;
char *scommt = SCOMMT;
char *ecommt = ECOMMT;
char *xcommt = XCOMMT;

static int
at H1(char const *,s)
    {
   char const *p = s;
   int c, d;

   while ((c = *p++) != EOS) {
       d = GPBC;
       if (d != c) {
     putback(d);
     p--;
     while (p != s) putback(*--p);
     return 0;
       }
   }
   return 1;
    }


/*  Definitions of diversion files.  The last 6 characters MUST be
    "XXXXXX" -- that is a requirement of mktemp().  The character
    '0' is to be replaced by the diversion number; we assume here
    that it is just before the Xs.  If not, you will have to alter
    the definition of UNIQUE.
*/

static char DIVNAM[] =
#if  unix
     "/usr/tmp/m40XXXXXX";
#else
#if  vms
     "sys$login:m40XXXXXX";
#else
#if  ibmvm
     "PDM4TMP.$0XXXXXX";
#else
     "M0XXXXXX"; /* must be no more than 8 letters */
#endif
#endif
#endif
int UNIQUE = sizeof DIVNAM - 7; /* where to change m4temp.     */
char *m4temp;       /* filename for diversions     */
extern char *mktemp();


static void
cantread H1(char const *,s)
    {
   error2("cannot read file", s);
    }


/*  initkwds()
    initialises the hash table to contain all the m4 built-in functions.
    The original version breached module boundaries, but there did not
    seem to be any benefit in that.
*/
static void
initkwds H0(void)
    {
   static int done = 0;
   register int i;
   static struct { char *name; int type; } keyword[] =
       {
         { "include",      INCLTYPE },
         { "sinclude",     SINCTYPE },
         { "define",       DEFITYPE },
         { "defn",         DEFNTYPE },
         { "divert",       DIVRTYPE },
         { "eval",         EXPRTYPE },
         { "substr",       SUBSTYPE },
         { "ifelse",       IFELTYPE },
         { "ifdef",        IFDFTYPE },
         { "len",          LENGTYPE },
         { "incr",         INCRTYPE },
         { "decr",         DECRTYPE },
         { "dnl",          DNLNTYPE },
         { "index",        INDXTYPE },
         { "changecom",    CHNCTYPE },
         { "changequote",  CHNQTYPE },
#ifdef EXTENDED
         { "paste",        PASTTYPE },
         { "spaste",       SPASTYPE },
         { "m4trim",   TRIMTYPE },
         { "defquote", DEFQTYPE },
         { "__file__", FILETYPE },
         { "m4getenv",     GENVTYPE },
#endif
         { "popdef",       POPDTYPE },
         { "pushdef",      PUSDTYPE },
         { "dumpdef",      DUMPTYPE },
         { "shift",        SHIFTYPE },
         { "translit",     TRNLTYPE },
         { "undefine",     UNDFTYPE },
         { "undivert",     UNDVTYPE },
         { "divnum",       DIVNTYPE },
         { "maketemp",     MKTMTYPE },
         { "errprint",     ERRPTYPE },
         { "m4wrap",       M4WRTYPE },
         { "m4exit",       EXITTYPE },
#if  unix
         { "unix",         MACRTYPE },
#endif
#if  msdos
         { "msdos", MACRTYPE },
#endif
#if  macos
         { "macos", MACRTYPE },
#endif
#if  vms
         { "vms",          MACRTYPE },
#endif
#if  ibmvm
         { "ibmvm",        MACRTYPE },
#endif
#if  mvs
         { "mvs",          MACRTYPE },
#endif
         { "syscmd",       SYSCTYPE },
         { "sysval",       SYSVTYPE },
         { (char*)0,   0        }
       };

   if (!done) {
       for (i = 0; keyword[i].type != 0; i++)
     addkywd(prefix, keyword[i].name, keyword[i].type);
       done = 1;
   }
    }


/*  inspect(Name)
    Build an input token.., considering only those which start with
    [A-Za-z_].  This is fused with lookup() to speed things up.
    name must point to an array of at least MAXTOK characters.
*/
static ndptr
inspect H1(char *,name)
    {
   static char tok2long[] = "identifier too long";
   register unsigned char *tp  = (unsigned char *)name;
   register unsigned char *etp = tp+(MAXTOK-1);
   register int c;
   register unsigned int h = 0;
   register ndptr p;

   for (;;) {
       if (compare == strcmp) {
     while (is_sym2(c = GPBC)) {
         if (tp == etp) error1(tok2long);
         HUPDATE(h, *tp++ = c);
     }
       } else {
     while (is_sym2(c = GPBC)) {
         if (tp == etp) error1(tok2long);
         HUPDATE(h, dncase[*tp++ = c]);
     }
       }
       *tp = EOS;
       for (p = hashtab[(int)(h%HASHSIZE)]; p != nil; p = p->nxtptr) {
     if (compare(name, p->name) == 0) {
         putback(c);
         return p;
     }
       }
#ifdef INFIXCHAR
       if (c == INFIXCHAR) {
     if (tp == etp) error1(tok2long);
     HUPDATE(h, *tp++ = c);
           continue;
       }
#endif
       putback(c);
       return nil;
   }
    }


/*   macro - the workhorse..
*/
static void
macro H0(void)
    {
   char token[MAXTOK];
   register int t;
   register FILE *op = active;
   static char ovmsg[] = "internal stack overflow";
   static char comsg[] = "unexpected EOF in comment";

   for (;;) {
       t = GPBC;
       if (is_sym1(t)) {
     register char *s;
     register ndptr p;

     putback(t);
     if ((p = inspect(s = token)) == nil) {
         if (sp < 0) {
        while ((t = *s++) != '\0')
            putc(t, op);
         } else {
        while ((t = *s++) != '\0')
            chrsave(t);
         }
     } else {
         /* real thing.. First build a call frame */
         if (sp >= STACKMAX-6) error1(ovmsg);
         mstack[1+sp].sfra = fp;      /* previous call frm */
         mstack[2+sp].sfra = p->type; /* type of the call  */
         mstack[3+sp].sfra = 0;    /* parenthesis level */
         fp = sp+3;          /* new frame pointer */
         /* now push the string arguments */
         mstack[4+sp].sstr = p->defn; /* defn string */
         mstack[5+sp].sstr = p->name; /* macro name  */
         mstack[6+sp].sstr = ep;      /* start next.. */
         sp += 6;

         t = GPBC;
         putback(t);
         if (t != LPAREN) { putback(RPAREN); putback(LPAREN); }
     }
       } else
       if (t == EOF) {
     inclose();
     if (ilevel < 0) break;
       } else
       /* non-alpha single-char token seen..
     [the order of else if .. stmts is important.]
       */
       if (t == lquote[0] && at(lquote+1)) { /* strip quotes */
     register int nlpar;

     for (nlpar = 1; ; ) {
         t = GPBC;
         if (t == rquote[0] && at(rquote+1)) {
        if (--nlpar == 0) break;
        strsave(rquote);
        continue;
         } else
         if (t == lquote[0] && at(lquote+1)) {
        nlpar++;
        strsave(lquote);
        continue;
         } else {
        if (t == vquote[0] && at(vquote+1)) {
            t = GPBC;
        }
        if (t == EOF) {
            error1("missing right quote");
        }
         }
         if (sp < 0) {
        putc(t, op);
         } else {
        chrsave(t);
         }
     }
       } else
       if (sp < 0) {         /* not in a macro at all */
     if (t != scommt[0] || !at(scommt+1)) {
         /* t doesn't begin a comment, so */
         putc(t, op);     /* copy it to output */
     } else
     if (strip) {         /* discard a comment */
         for (;;) {
        t = GPBC;
        if (t == EOF) error1(comsg);
        if (t == ecommt[0] && at(ecommt+1)) break;
         }
     } else {       /* copy comment to output */
         prstr(op, scommt);
         for (;;) {
        t = GPBC;
        if (t == EOF) error1(comsg);
        if (t == ecommt[0] && at(ecommt+1)) break;
        putc(t, op);
         }
         prstr(op, ecommt);
     }
       } else
       switch (t) {
     /*  There is a peculiar detail to notice here.
         Layout is _always_ discarded after left parentheses,
         but it is only discarded after commas if they separate
         arguments.  For example,
         define(foo,`|$1|$2|')
         foo( a, b)    => |a|b|
         foo(( a ), ( b ))   => |(a )|(b )|
         foo((a, x), (b, y)) => |(a, x)|(b, y)|
         I find this counter-intuitive, and would expect the code
         for LPAREN to read something like this:

         if (PARLEV == 0) {
        (* top level left parenthesis: skip layout *)
        do t = GPBC; while (is_blnk(t));
        putback(t);
         } else {
        (* left parenthesis inside an argument *)
        chrsave(t);
         }
         PARLEV++;

         However, it turned out that Oz wrote the actual code
         very carefully to mimic the behaviour of "real" m4;
         UNIX m4 really does skip layout after all left parens
         but only some commas in just this fashion.  Sigh.
     */
     case LPAREN:
         if (PARLEV > 0) chrsave(t);
         do t = GPBC; while (is_blnk(t));   /* skip layout */
         putback(t);
         PARLEV++;
         break;

     case COMMA:
         if (PARLEV == 1) {
        chrsave(EOS);     /* new argument   */
        if (sp >= STACKMAX) error1(ovmsg);
        do t = GPBC; while (is_blnk(t)); /* skip layout */
        putback(t);
        mstack[++sp].sstr = ep;
         } else {
        chrsave(t);
         }
         break;

     case RPAREN:
         if (--PARLEV > 0) {
        chrsave(t);
         } else {
        stae *argv = &mstack[fp+1];
        int   argc = sp-fp;

        chrsave(EOS);     /* last argument */
        if (sp >= STACKMAX) error1(ovmsg);
#ifdef  DEBUG
        fprintf(stderr, "argc = %d\n", argc);
        for (t = 0; t < argc; t++)
            fprintf(stderr, "argv[%d] = %s\n", t, arg(t));
#endif
        /*  If argc == 3 and arg(2) is null, then we
            have a call like `macro_or_builtin()'.  We
            adjust argc to avoid further checking..
        */
        if (argc == 3 && !arg(2)[0]) argc--;

        switch (CALTYP) {
            case MACRTYPE:
           expand(argv, argc);
           break;

            case DEFITYPE:      /* define(..) */
           for (; argc > 2; argc -= 2, argv += 2)
               dodefine(arg(2), argc > 3 ? arg(3) : null);
           break;

            case PUSDTYPE:      /* pushdef(..) */
           for (; argc > 2; argc -= 2, argv += 2)
               dopushdef(arg(2), argc > 3 ? arg(3) : null);
           break;

            case DUMPTYPE:
           dodump(argv, argc);
           break;

            case EXPRTYPE:      /* eval(Expr) */
           {   /* evaluate arithmetic expression */
               /* eval([val: 0[, radix:10 [,min: 1]]]) */
               /* excess arguments are ignored */
               /* eval() with no arguments returns 0 */
               /* this is based on V.3 behaviour */
               int min_digits = 1;
               int radix = 10;
               long int value = 0;

               switch (argc) {
              default:
                  /* ignore excess arguments */
                  /*FALLTHROUGH*/
              case 5:
                  min_digits = expr(arg(4));
                  /*FALLTHROUGH*/
              case 4:
                  radix = expr(arg(3));
                  /*FALLTHROUGH*/
              case 3:
                  value = expr(arg(2));
                  /*FALLTHROUGH*/
              case 2:
                  break;
               }
               pbrad(value, radix, min_digits);
           }
           break;

            case IFELTYPE:      /* ifelse(X,Y,IFX=Y,Else) */
           doifelse(argv, argc);
           break;

            case IFDFTYPE:      /* ifdef(Mac,IfDef[,IfNotDef]) */
           /* select one of two alternatives based on the existence */
           /* of another definition */
           doifdef(argv, argc);
           break;

            case LENGTYPE:      /* len(Arg) */
           /* find the length of the argument */
           pbnum(argc > 2 ? strlen(arg(2)) : 0);
           break;

            case INCRTYPE:      /* incr(Expr) */
           /* increment the value of the argument */
           if (argc > 2) pbnum(expr(arg(2)) + 1);
           break;

            case DECRTYPE:      /* decr(Expr) */
           /* decrement the value of the argument */
           if (argc > 2) pbnum(expr(arg(2)) - 1);
           break;

            case SYSCTYPE:      /* syscmd(Command) */
           /* execute system command */
           /* Make sure m4 output is NOT interrupted */
#if  HAS_SYSCMD > 0
           flushall();
           if (argc > 2) sysval = system(arg(2));
#else
           error2("not implemented:", arg(1));
#endif
           break;

            case SYSVTYPE:      /* sysval() */
           /* return value of the last system call.  */
           pbnum(sysval);
           break;

            case INCLTYPE:      /* include(File) */
           for (t = argc; --t >= 2; )
               if (!incopen(arg(t))) cantread(arg(t));
           break;

            case SINCTYPE:      /* sinclude(File) */
           for (t = argc; --t >= 2; )
               (void) incopen(arg(t));
           break;

#ifdef EXTENDED
            case PASTTYPE:      /* paste(File) */
           for (t = 2; t < argc; t++)
               if (!dopaste(arg(t))) cantread(arg(t));
           break;

            case SPASTYPE:      /* spaste(File) */
           for (t = 2; t < argc; t++)
               (void) dopaste(arg(t));
           break;

            case TRIMTYPE:      /* m4trim(Source,..) */
           if (argc > 2) m4trim(argv, argc);
           break;

            case DEFQTYPE:      /* defquote(Mac,...) */
           dodefqt(argv, argc);
           break;

            case QUTRTYPE:      /* <quote>(text...) */
           doqutr(argv, argc);
           break;

            case FILETYPE:      /* __file__ */
           strsave(inname[ilevel]);
           break;

            case GENVTYPE:      /* m4getenv(Var,Def) */
           if (argc > 2) {
               char *v = getenv(arg(2));
               if (v == NULL && argc > 3)
              v = arg(3);
               if (v == NULL)
              error2("no environment var", arg(2));
               pbqtd(v);
           } else {
               error2("no arguments:", arg(1));
           }
           break;

#endif

            case CHNQTYPE:      /* changequote(L,R,V) */
           dochq(argv, argc);
           break;

            case CHNCTYPE:      /* changecom(L,R) */
           dochc(argv, argc);
           break;

            case SUBSTYPE:      /* substr(S,O,L,B) */
           /* select substring */
           if (argc > 3) dosub(argv, argc);
           break;

            case SHIFTYPE:      /* shift(~args~) */
           /* push back all arguments except the first one */
           /* (i.e.  skip arg(2)) */
           if (argc > 3) {
               for (t = argc-1; t > 3; t--) {
              pbqtd(arg(t));
              putback(',');
               }
               pbqtd(arg(3));
           }
           break;

            case DIVRTYPE:      /* divert(N) */
           if (argc > 2 && (realdiv = expr(arg(2))) != 0) {
               dodiv(realdiv);
           } else {
               active = stdout;
               oindex = 0;
               realdiv = 0;
           }
           op = active;
           break;

            case UNDVTYPE:      /* undivert(N...) */
           /* This does not change the current */
           /* diversion at all.  */
           doundiv(argv, argc);
           op = active;
           break;

            case DIVNTYPE:      /* divnum() */
           /* return the number of current output diversion */
           pbnum(realdiv);
           break;

            case UNDFTYPE:      /* undefine(..) */
           /* undefine a previously defined macro(s) or m4 keyword(s). */
           for (t = 2; t < argc; t++) remhash(arg(t), ALL);
           break;

            case POPDTYPE:      /* popdef(Mac...) */
           /* remove the topmost definitions of macro(s) or m4 keyword(s). */
           for (t = 2; t < argc; t++) remhash(arg(t), TOP);
           break;

            case MKTMTYPE:      /* maketemp(Pattern) */
           /* create a temporary file */
           if (argc > 2) pbstr(mktemp(arg(2)));
           break;

            case TRNLTYPE:      /* translit(Source,Dom,Rng) */
           /* replace all characters in the source string that */
           /* appears in the "from" string with the corresponding */
           /* characters in the "to" string. */

           if (argc > 2) {
               translit(arg(2),
              (argc > 3 ? arg(3) : null),
                   (argc > 4 ? arg(4) : null),
                   argc > 5);
               pbstr(arg(2));
           }
           break;

            case INDXTYPE:      /* index(Source,Target) */
           /* find the index of the second argument string in */
           /* the first argument string. -1 if not present. */
           pbnum(argc > 3 ? indx(arg(2), arg(3)) : -1);
           break;

            case ERRPTYPE:      /* errprint(W,...,W) */
           /* print the arguments to stderr file */
           for (t = 2; t < argc; t++) fprintf(stderr, "%s ", arg(t));
           fprintf(stderr, "\n");
           break;

            case DNLNTYPE:      /* dnl() */
           /* eat upto and including newline */
           while ((t = GPBC) != '\n' && t != EOF) {}
           break;

            case M4WRTYPE:      /* m4wrap(AtExit) */
           /* set up for wrap-up/wind-down activity.   */
           /* NB: if there are several calls to m4wrap */
           /* only the last is effective; strange, but */
           /* that's what System V does.               */
           if (m4wraps != null) free(m4wraps);
           m4wraps = argc > 2 ? dupstr(arg(2)) : null;
           break;

            case EXITTYPE:      /* m4exit(Expr) */
           /* immediate exit from m4.  */
           m4exit(argc > 2 ? expr(arg(2)) : 0);
           break;

            case DEFNTYPE:      /* defn(Mac) */
           for (t = 2; t < argc; t++)
               dodefn(arg(t));
           break;

            default:
           error1("major botch in eval.");
           break;
        }

        ep = PREVEP;      /* flush strspace */
        sp = PREVSP;      /* previous sp..  */
        fp = PREVFP;      /* rewind stack... */
         }
         break;

     default:
         chrsave(t);         /* stack the char */
         break;
       }
   }
    }


#if macos
#define main MAIN
#endif

int
main H2(int,argc, char **,argv)
    {
   register int c;
   register int n;
   char *p;

#ifdef  SIGINT
   if (signal(SIGINT, SIG_IGN) != SIG_IGN)
     signal(SIGINT, onintr);
#endif
   for (n = 0; n < HASHSIZE; n++) hashtab[n] = nil;
   for (n = 0; n < MAXOUT; n++) outfile[n] = NULL;
   m4wraps = null;
   init_chtype();

   while ((c = getopt(argc, argv, "celVPM:D:U:o:B:H:S:T:")) != EOF) {
       switch (c) {
     case 'V':
         fprintf(stderr, "1997.11.20 public domain m4\n");
         exit(0);

#if 0
     case 's':      /* enable #line sync in output */
         error1("this version does not support -s");
#endif

     case 'c':      /* strip comments */
         strip ^= 1;
         break;

     case 'e':      /* interactive */
         (void) signal(SIGINT, SIG_IGN);
         setbuf(stdout, NULL);
         break;

#ifdef  EXTENDED
     case 'l':      /* lowercase letters in lookup */
         compare = dncmp;
         break;
#endif

     case 'M':      /* Macro prefix */
         prefix = optarg;
         break;

     case 'P':      /* force m4_ prefix on everything */
         prefix = "m4_";
         break;

     case 'D':               /* define something..*/
         strcpy(strspace, optarg);
         for (p = strspace; *p != '\0' && *p != '='; p++) {}
         if (*p != '\0') *p++ = EOS;
         dodefine(strspace, p);
         break;

     case 'U':               /* undefine...       */
         /* We had better install the keywords now, */
         /* if that hasn't already been done, so the */
         /* user can delete some of them. */
         initkwds();
         remhash(optarg, TOP);
         break;

     case 'B': case 'H':  /* System V compatibility */
     case 'S': case 'T':  /* ignore them */
         break;

     case 'o':      /* specific output   */
         if (!freopen(optarg, "w", stdout)) {
        perror(optarg);
        m4exit(EXIT_FAILURE);
         }
         break;

     case '?':
     default:
        usage();
       }
   }

   initkwds();         /* install bims now maybe    */
   active = stdout;    /* default active output     */
   m4temp = mktemp(DIVNAM);  /* filename for diversions   */

   sp = -1;         /* stack pointer initialized */
   fp = 0;          /* frame pointer initialized */

   if (optind == argc) {     /* no more args; read stdin  */
       if (!incopen("-")) cantread("standard input");
       macro();        /* process that file         */
   } else           /* file names in commandline */
   for (; optind < argc; optind++) {
       if (!incopen(argv[optind])) cantread(argv[optind]);
       macro();
   }

   if (*m4wraps) {     /* anything for rundown ??   */
       ilevel = 0;        /* in case m4wrap includes.. */
       infile[ilevel] = ifile = 0; /* ensure no more input      */
       inname[ilevel] = "-"; /* treat as empty stdin      */
       pbstr(m4wraps);       /* user-defined wrapup act   */
       macro();        /* last will and testament   */
   } else {         /* default wrap-up: undivert */
       for (n = 1; n < MAXOUT; n++)
     if (outfile[n] != NULL) getdiv(n);
   }

   if (outfile[0] != NULL) { /* remove bitbucket if used  */
       (void) fclose(outfile[0]);
       m4temp[UNIQUE] = '0';
       (void) remove(m4temp);
   }
   m4exit(EXIT_SUCCESS);
   return 0;
    }

