/*  File   : misc.c
    Author : Ozan Yigit
    Editor : Richard A. O'Keefe
    Updated: 11/20/97
    Purpose: Miscellaneous support code for PD M4.
*/
#include "mdef.h"
#include "extr.h"
#include "ourlims.h"
#include "os.h"

/* Modification History:
 *
 * Nov 20 1997 RAOK Fixed incopen() when HAS_SYSCMD == 1;
 *      use tmpnam() to generate file names, not m4temp.
 *
 * Nov 19 1997 RAOK Modified DOS/VMS version of getopt().
 *      Now it will acccept a leading / as well
 *      as a leading - for an option, and it
 *      will skip : or = after an option that
 *      has an argument, so you can write
 *          M4 /O=FOO.OUT
 *      as well as
 *          m4 -o foo.out
 *      Added flushall().
 *      Most importantly, added incopen(), inclose().
 *
 * Nov 17 1997 RAOK Renamed error -> error1; added error2.
 *      Moved "m4: " text into these functions.
 *
 * Nov 12 1997 RAOK     Implemented mktemp() for Think C.  See the
 *      comments; it's actually pretty disgusting.
 *
 * Nov 11 1997 RAOK Use new os.h header and use ANSI remove()
 *      function instead of UNIX unlink().
 *      The major change is having arbitrary-length
 *      strings for lquote, rquote, and vquote.
 *
 * Nov  7 1997 RAOK dupstr() now reports an error and exits if
 *      there isn't enough memory to allocate the new
 *      string.  Previously it returned a null pointer,
 *      but the rest of the program wasn't prepared to
 *      deal with that.  This isn't the best way to
 *      handle memory exhaustion, but it's a lot better
 *      than quietly going crazy.
*/


#if macos | ibmvm

/*  mktemp(string)
    replaces the last six Xs in string with something magical to make a
    unique file name.  That's the idea, anyway.  In UNIX systems, it
    would replace the last five Xs with the process ID in decimal, and
    the X before those with a unique letter.  This means that mktemp()
    could only generate 26 unique names, and of course it has problems
    over a network, where processes on different machines could have the
    same process ID.  This is why the C standard uses tmpnam() instead,
    but that would mean more of a rewrite than I feel inclined for just
    at the moment.  It is worth noting that Think C's tmpnam() just
    generates the sequence temp0001 temp0002 ..., which is pretty much
    useless in a Multifinder environment.

    What I've decided to do is a pretty disgusting crock.  I rely on
    Mac processes all living in the same address space, so that a
    static variable in one program can't have the same address as
    a static variable in another program.  Shift the address right 2,
    and you have 30 bits, which fits neatly into 6 Xs using base 32.
    To permit multiple uses by the same program, I take the value of
    the counter and xor it in.  Ok, this is pretty awful, but it's
    _better_ than Think C's tmpnam().
*/

char *
mktemp H1(char *,s)
    {
   char *b, *e, *p;    /* point into s */
   static unsigned long counter = 0;
   unsigned long bits;
   static char base32[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ012345";
   FILE *probe;
   int limit;       /* sanity check */

   for (e = s; *e; e++) {};
   for (b = e; b != s && b[-1] == 'X'; b--) {}
   if (e-b > 6) b = e-6;
   /* Now [b..e) is the last six Xs of s */
   /* if there are fewer Xs, this segment has all of them */

   for (limit = 0; limit < 1000; limit++) {
       p = e;
       bits = ((unsigned long)&counter >> 2) ^ counter++;
       while (p != b) *--p = base32[bits&31], bits >>= 5;
       probe = fopen(s, "r");
       if (probe == NULL) return;
       fclose(probe);
   }
   error2("1000 mktemp attempts failed:", s);
    }

#endif


#ifdef  DUFFCP

/*  This version of the ANSI standard function memcpy()
    uses Duff's Device (tm Tom Duff)  to unroll the copying loop:
   while (count-- > 0) *to++ = *from++;
*/
generic *
memcpy H3(
    generic       *,from,
    generic const *,to,
    int            ,count
)
    {
   register generic       *f = (generic       *)from;
   register generic const *t = (generic const *)to;

   if (count > 0) {
       register int loops = (count+8-1) >> 3;  /* div 8 round up */

       switch (count & (8-1)) { /* mod 8 */
           case 0: do {   *t++ = *f++;
     case 7:     *t++ = *f+;
     case 6:     *t++ = *f++;
     case 5:     *t++ = *f++;
     case 4:     *t++ = *f++;
     case 3:     *t++ = *f++;
     case 2:     *t++ = *f++;
     case 1:     *t++ = *f++;
             } while (--loops > 0);
       }
   }
    }

#endif


char memsg[] = "not enough memory";

/*  dupstr(s)
    return a new malloc()ed copy of s -- same as V.3's strdup().
*/
char *
dupstr H1(char const *,s)
    {
   register int n = strlen(s)+1;
   char *p = malloc(n);

   if (p == (char*)0) error1(memsg);
   (void)memcpy(p, s, n);
   return p;
    }


/*  indx(s1, s2)
    if s1 can be decomposed as alpha || s2 || omega, return the length
    of the shortest such alpha, otherwise return -1.  This is essentially
    the same as the ANSI C strstr() function.  We should consider adapting
    Doug Gwyn's public domain implementation of strstr().
*/
int
indx H2(char const *,s1, char const *,s2)
    {
   register char const *t;
   register char const *m;
   register char const *p;

   for (p = s1; *p; p++) {
       for (t = p, m = s2; *m && *m == *t; m++, t++);
       if (!*m) return p-s1;
   }
   return -1;
    }


char pbmsg[] = "too many characters pushed back";

/*  pbstr(s)
    push string s back onto the input stream.
    putback() has been unfolded here to improve performance.
    Example:
   s = <ABC>
   bp = <more stuff>
    After the call:
   bp = <more stuffCBA>
    It would be more efficient if we ran the pushback buffer in the
    opposite direction
*/
void
pbstr H1(register char const *,s)
    {
   register char const *es;
   register char       *zp;

   zp = bp;
   for (es = s; *es; ) es++; /* now es points to terminating NUL */
   bp += es-s;         /* advance bp as far as it should go */
   if (bp >= endpbb) error1(pbmsg);
   while (es > s) *zp++ = *--es;
    }


/*  ats(delimiter+1, rest_of_string)
    is used to match multicharacter delimiters.
   c = *es++;
   if (c == delim[0] && (np = ats(delim+1, es)) != 0) ... es = np;
    will advance es over a multicharacter delimiter (if it matches) or
    a single character (if it doesn't).  It's like at() in main.c.
*/
static char const *
ats H2(register char const *,s, register char const *,t)
    {
   char c;

   while ((c = *s++) != EOS)
       if (*t++ != c) return (char*)0;
   return t;
    }


/*  pbqtd() has a nasty problem with pushing back multicharacter
    delimiters:  the string to be pushed back and the string it is
    to be pushed into run in opposite directions.  There are five places
    where we have to do this, so it is worth haing a function just to
    get this right.
   zp = pbdelim(zp, d)
    moves the characters of d to zp, in reverse.
*/
static char *
pbdelim H2(char *,zp, char const *,d)
    {
   char const *s;

   for (s = d; *s != EOS; s++) {}
   while (s != d) *zp++ = *--s;
   return zp;
    }


/*  pbqtd(s)
    pushes string s back "quoted", doing whatever has to be done to it to
    make sure that the result will evaluate to the original value.  As it
    happens, we have only to add lquote and rquote.

    What should happen if the left and right quote characters in the
    string are not balanced?  It is for this that we added the vquote
    character.  We have to make a pass over the string anyway to find
    the end, so we might as well check for balance at the same time.
    If the string is not balanced, we put a vquote in front of each
    left and right quote.  If it is balanced, we don't.  We treat a
    string containing a vquote already as unbalanced; that isn't really
    necessary, but it simplifies the code.

*/
void
pbqtd H1(register char const *,s)
    {
   register char const *es;
   register char       *zp;
   register char        c;
   char const *np;        /* next pointer after delimiter */
   int parct;       /* "parenthesis" count */
   int vqtct;       /* number of vquotes needed */
   int isbal;       /* true if balanced */

   parct = 0;       /* no quotes seen yet */
   vqtct = 0;
   isbal = strcmp(lquote, rquote); /* never balanced if quotes equal */
   for (es = s; (c = *es++) != EOS; ) {
       if (c == rquote[0] && (np = ats(rquote+1, es)) != 0) {
     vqtct++;
     if (--parct < 0) isbal = 0;
     es = np;
       } else
       if (c == lquote[0] && (np = ats(lquote+1, es)) != 0) {
     vqtct++;
     parct++;
     es = np;
       } else
       if (c == vquote[0] && (np = ats(vquote+1, es)) != 0) {
     vqtct++;
     isbal = 0;
     es = np;
       }
   }
   es--;
   if (parct != 0) isbal = 0;

   /* Now es points to the terminating NUL and isbal is set */

   if (isbal) {        /* use old code */

       zp = bp;
       bp += 2+(es-s);    /* advance bp as far as it should go */
       if (bp >= endpbb) error1(pbmsg);
       zp = pbdelim(zp, rquote);
       while (es > s) *zp++ = *--es;
       zp = pbdelim(zp, lquote);

   } else {

       zp = bp;
       bp += 2+(es-s)+vqtct*strlen(vquote);
       if (bp >= endpbb) error1(pbmsg);
       zp = pbdelim(zp, rquote);
       while (es > s) {
     c = *--es;
     *zp++ = c;
     if ((c == rquote[0] && ats(rquote+1, es) != 0)
            || (c == lquote[0] && ats(lquote+1, es) != 0)
            || (c == vquote[0] && ats(vquote+1, es) != 0)
     ) {
         zp = pbdelim(zp, vquote);
     }
       }
       zp = pbdelim(zp, lquote);
   }

    }


void
prstr H2(register FILE *,f, register char const *,s)
    {
   int c;

   while ((c = *s++) != EOS) putc(c, f);
    }


/*  prqtd(f, s)
    writes string s "quoted" to output stream f.
    This uses the same quoting algorithm as pbqtd.
*/
void
prqtd H2(register FILE *,f, register char const *,s)
    {
   register char const *es;
   register char c;
   char const *np;        /* next pointer after delimiter */
   int parct;       /* "parenthesis" count */
   int vqtct;       /* number of vquotes needed */
   int isbal;       /* true if balanced */

   parct = 0;       /* no quotes seen yet */
   vqtct = 0;
   isbal = strcmp(lquote, rquote); /* never balanced if quotes equal */
   for (es = s; (c = *es++) != EOS; ) {
       if (c == rquote[0] && (np = ats(rquote+1, es)) != 0) {
     vqtct++;
     if (--parct < 0) isbal = 0;
     es = np;
       } else
       if (c == lquote[0] && (np = ats(lquote+1, es)) != 0) {
     vqtct++;
     parct++;
     es = np;
       } else
       if (c == vquote[0] && (np = ats(vquote+1, es)) != 0) {
     vqtct++;
     isbal = 0;
     es = np;
       }
   }
   es--;
   if (parct != 0) isbal = 0;

   /* Now es points to the terminating NUL and isbal is set */

   prstr(f, lquote);
   if (isbal) {
       for (es = s; (c = *es++) != EOS; )
     putc(c, f);
   } else {
       for (es = s; (c = *es++) != EOS; ) {
     if ((c == rquote[0] && ats(rquote+1, es) != 0)
      || (c == lquote[0] && ats(lquote+1, es) != 0)
      || (c == vquote[0] && ats(vquote+1, es) != 0)
     ) {
         prstr(f, vquote);
     }
     putc(c, f);
       }
   }
   prstr(f, rquote);

    }


/*  pbnum(n)
    convert a number to a (decimal) string and push it back.
    The original definition did not work for MININT; this does.
*/
void
pbnum H1(int,n)
    {
   register int num;

   num = n > 0 ? -n : n;  /* MININT <= num <= 0 */
   do {
       putback('0' - (num % 10));
   } while ((num /= 10) < 0);
   if (n < 0) putback('-');
    }


/*  pbrad(n, r, m)
    converts a number n to base r ([-36..-2] U [2..36]), with at least
    m digits.  If r == 10 and m == 1, this is exactly the same as pbnum.
    However, this uses the function int2str() from R.A.O'Keefe's public
    domain string library, and puts the results of that back.
    To be compatible with Unix System V Release 3's m4, we treat radix
    1 like this:
   push back abs(n) "1"s, then
   if abs(n) < m, push back m-abs(n) "0"s, then
   if n < 0, push back one "-".
    I am not really happy about devoting so much code to a freak.
    However, UNIX System V ignores negative or zero radices, producing
    results in radix 10.  We have a use for negative radices in the
    range -36..-2, so there remains an incompatibility.  We also have
    a use for radix 0:  unsurprisingly, it is related to radix 0 in
    Edinburgh Prolog.
   eval('c1c2...cn', 0, m)
    pushes back max(m-n,0) blanks and the characters c1...cn.  This can
    adjust to any byte size as long as UCHAR_MAX = (1 << CHAR_BIT) - 1.
    In particular, eval(c, 0) where 0 < c <= UCHAR_MAX, pushes back the
    character with code c.  Note that this has to agree with eval(); so
    both of them have to use the same byte ordering.
*/
void
pbrad H3(
    long int,n,
    int     ,r,
    int     ,m
)
    {
   char buffer[LONG_BIT + 8];
   char *p;
   int L;

   if (r == 0) {
       unsigned long int x = (unsigned long)n;
       int k;

       for (k = 0; x; x >>= CHAR_BIT, k++) buffer[k] = x & UCHAR_MAX;
       for (L = k; --L >= 0; ) putback(buffer[L]);
       for (L = m-k; --L >= 0; ) putback(' ');
       return;
   } else
   if (r == 1) {
       if (n < 0) {
     if ((m += n) < 0) m = 0;
     do { putback('1'); n++; } while (0 != n);
     while (m != 0) { putback('0'); m--; }
     putback('-');
       } else {
     if ((m -= n) < 0) m = 0;
     while (n != 0) { putback('1'); n--; }
     while (m != 0) { putback('0'); m--; }
       }
       return;
   } else
   if (r < -36 || r > 36 || r == -1) {
       r = 10;
   }
   L = m - (int2str(p = buffer, -r, n)-buffer);
   if (buffer[0] == '-') L++, p++;
   if (L > 0) {
       pbstr(p);
       while (--L >= 0) putback('0');
       if (p != buffer) putback('-');
   } else {
       pbstr(buffer);
   }
    }


char csmsg[] = "string space overflow";

/*  strsave(s)
    put the characters of the string s in the string space
*/
void
strsave H1(char const *,s)
    {
   int c;

   if (sp < 0) {
       while ((c = *s++) != EOS) putc(c, active);
   } else {
       while ((c = *s++) != EOS) {
     if (ep < endest) *ep++ = c;
     else error1(csmsg);
       }
   }
    }


/*  getdiv(ind)
    read in a diversion file and then delete it.
*/
void
getdiv H1(int,ind)
    {
   register int c;
   register FILE *dfil;
   register FILE *afil;

   afil = active;
   if (outfile[ind] == afil)
       error1("undivert: diversion still active.");
   if (outfile[ind] == NULL)
       return;
   (void) fclose(outfile[ind]);
   outfile[ind] = NULL;
   m4temp[UNIQUE] = '0' + ind;
   if ((dfil = fopen(m4temp, "r")) == NULL)
       error2("cannot undivert", m4temp);
   while ((c = getc(dfil)) != EOF) putc(c, afil);
   (void) fclose(dfil);
   if (remove(m4temp)) error2("cannot remove", m4temp);
    }


/*  flushall()
    flushes all partial output buffers.  In ANSI/ISO C, this is easy,
    as fflush(0) has been defined to do that.  Otherwise, we are in
    the lucky position of knowing what the output files all are.
*/
void
flushall H0(void)
    {
#if  __STDC__
   fflush((FILE*)0);
#else
   int i;

   fflush(stdout);
   fflush(stderr);
   for (i = 0; i < MAXOUT; i++)
       if (outfile[i] != NULL)
     fflush(outfile[i]);
#endif
    }


/*  m4exit(status)
    frees all resources held by the program and exits.
    In this case, freeing resources means deleting all the temporary
    files that were created to hold diversions; temporary files created
    by system() commands are not deleted because m4 itself doesn't know
    about them.
*/
void
m4exit H1(int,status)
    {
   register int n;

   for (n = 0; n < MAXOUT; n++) {
       if (outfile[n] != NULL) {
     (void) fclose(outfile[n]);
     m4temp[UNIQUE] = '0' + n;
     (void) remove(m4temp);
       }
   }
   exit(status);
    }


/*  error2(s, t)
    close all files, report a fatal error, and quit, letting the caller know.
*/
void
error2 H2(char const *,s, char const *,t)
    {
   fprintf(stderr, "m4: %s", s);
   if (t != NULL) fprintf(stderr, " %s\n", t);
   fprintf(stderr, "\n");
   m4exit(EXIT_FAILURE);
    }

void
error1 H1(char const *,s)
    {
   error2(s, (char*)NULL);
    }


/*  Interrupt handling
*/
static char *msg = "\ninterrupted.";

/*ARGSUSED*/
void
onintr H1(int,signo)
    {
   error1(msg);
    }


/*  This function is in the wrong file.
    main() and usage() should be in the same file,
    and macro() should be somewhere else.
*/
void
usage H0(void)
    {
   fprintf(stderr, "Usage: m4 %s \\\n          %s file...\n",
       "[-e] [-c] [-l] [-V] [-P] [-Mprefix] [-[BHST]n]",
       "[-Dname[=val]]... [-Uname]...");
   m4exit(EXIT_FAILURE);
    }


#ifdef GETOPT
/* Henry Spencer's getopt() - get option letter from argv */

char *optarg;       /* Global argument pointer. */
int optind = 0;        /* Global argv index. */

static char *scan = NULL; /* Private scan pointer. */

#ifndef __STDC__
extern  char *index();
#define strchr index
#endif

#if vms || msdos
#define is_opt_mark(c) ((c) == '-' || (c) == '/')
#define is_arg_mark(c) ((c) == ':' || (c) == '=')
#else
#define is_opt_mark(c) ((c) == '-')
#define is_arg_mark(c) (0)
#endif
int
getopt H3(
    int          const,argc,
    char *const *const,argv,
    char  const *const,optstring
)
    {
   register char c;
   register char *place;

   optarg = NULL;

   if (scan == NULL || *scan == EOS) {
       if (optind == 0) optind++;
       if (optind >= argc
        || !is_opt_mark(argv[optind][0])
        || argv[optind][1] == EOS)
     return EOF;
       if (strcmp(argv[optind], "--") == 0) {
     optind++;
     return EOF;
       }
       scan = argv[optind]+1;
       optind++;
   }
   c = *scan++;
   place = strchr(optstring, c);

   if (place == NULL || c == ':') {
       fprintf(stderr, "%s: unknown option -%c\n", argv[0], c);
       return '?';
   }
   place++;
   if (*place == ':') {
       if (*scan != EOS) {
     if (is_arg_mark(*scan)) scan++;
     optarg = scan;
     scan = NULL;
       } else {
     optarg = argv[optind];
     optind++;
       }
   }
   return c;
    }
#endif



#if HAS_SYSCMD == 1

/* How many characters should we allocate for a scratch file name? */
#ifdef  L_tmpnam
#define SCRATCH_MAX L_tmpnam
#else
#ifdef  FILENAME_MAX
#define SCRATCH_MAX FILENAME_MAX
#else
#define SCRATCH_MAX 80
#endif
#endif
#endif

/*  incopen(filename)
    handles three cases:
   filename = "-"      => use stdin
   filename = "|"command  => popen() command
   otherwise     => fopen() filename
*/
int
incopen H1(char const *,filename)
    {
   FILE *f;

   if (ilevel == MAXINP-1) error2("includes nested too deep",filename);
   if (filename[0] == '-' && filename[1] == '\0') {
       f = stdin;
#if  HAS_SYSCMD == 2
   } else
   if (filename[0] == '|') {
       extern FILE *popen P2(char const *, char const *);
       f = popen(filename+1, "r");
       sysval = f == NULL ? -1 : 0;
#endif
#if  HAS_SYSCMD == 1
   } else
   if (filename[0] == '|') {
       static char scratchname[SCRATCH_MAX];
       char *command = (char *)malloc(
        strlen(filename+1) +    /* command */
        strlen(" >") +       /* redirection */
        sizeof scratchname);       /* output file */

       if (NULL == command) error1(memsg);
       (void) tmpnam(scratchname);
       sprintf(command, "%s >%s", filename+1, scratchname);
       sysval = system(command);         /* run it */
       free(command);
       filename = m4temp;
       f = fopen(filename, "r");
#endif
   } else {
       f = fopen(filename, "r");
   }
   if (f == (FILE*)0) return 0;
   ilevel++;
   infile[ilevel] = ifile = f;
   inname[ilevel] = dupstr(filename);
   bbstack[ilevel] = bb;
   bb = bp;
   return 1;
    }

void
inclose H0(void)
    {
   char *n = inname[ilevel];

   if (n[0] == '-' && n[1] == '\0') {
       ;
    #if HAS_SYSCMD == 2
   } else
   if (n[0] == '|') {
       extern int pclose P1(FILE *);
       pclose(ifile);
    #endif
   } else {
       fclose(ifile);
   }
   free(inname[ilevel]);
   bb = bbstack[ilevel];
   --ilevel;
   if (ilevel >= 0) ifile = infile[ilevel];
    }

